<?php

/**
 * Nonce abstraction for protecting forms against cross-site request forgery
 *
 * @package Caldera_Forms Modified by QuantumCloud
 * @author    Josh Pollock <Josh@CalderaWP.com>
 * @license   GPL-2.0+
 * @link
 * @copyright 2016 CalderaWP LLC
 */
class Qcformbuilder_Forms_Render_Nonce {

	/**
	 * Nonce action prefix
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected static $action = 'qcformbuilder_forms_front_';

	/**
	 * Create verification nonce
	 *
	 * @since 1.5.0
	 *
	 * @param string $form_id Form ID
	 *
	 * @return string
	 */
	public static function create_verify_nonce( $form_id ){
		return wp_create_nonce( self::nonce_action( $form_id ) );
	}

	/**
	 * Get name of nonce field
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public static function nonce_field_name( $form_id = false ){
		$name = '_wfb_verify';
		if( $form_id ){
			$name .= '_' . $form_id;
		}

		return $name;
	}

	/**
	 * Verify the verification nonce
	 *
	 * @since 1.5.0
	 *
	 * @param string $nonce Nonce to check
	 * @param string $form_id Form ID
	 *
	 * @return false|int
	 */
	public static function verify_nonce( $nonce, $form_id ){
		$valid = wp_verify_nonce( $nonce, self::nonce_action( $form_id ) );
		if( ! $valid ){
			/**
			 * Fires when form submission is stopped by invalid security token
			 *
			 * @since 1.5.0
			 *
			 * @param string $form_id ID of form that the
			 */
			do_action( 'qcformbuilder_forms_verification_token_failed', $form_id );
		}
		return $valid;
	}

	/**
	 * Create nonce field for use in form
	 *
	 * @since 1.5.0
	 *
	 * @param $form_id
	 *
	 * @return string
	 */
	public static function nonce_field( $form_id, $from_esi = false ){

		if ( ! $from_esi ) {
			if ( method_exists( 'LiteSpeed_Cache_API', 'esi_enabled' ) && LiteSpeed_Cache_API::esi_enabled() ) {
				if ( method_exists( 'LiteSpeed_Cache_API', 'v' ) && LiteSpeed_Cache_API::v( '1.2.4' ) ) {
					$params = array( 'form_id' => $form_id ) ;
					return LiteSpeed_Cache_API::esi_url( 'qcformbuilder_forms', 'Qcformbuilder Forms', $params ) ;
				}
			}
		}

		$nonce_field = '<input type="hidden" id="' . esc_attr( self::nonce_field_name( $form_id ) ) . '" name="' . esc_attr( self::nonce_field_name() ) . '" value="' . esc_attr( self::create_verify_nonce( $form_id ) ) . '"  data-nonce-time="' . esc_attr( time() ) . '" />';
		$nonce_field .= wp_referer_field( false );
		return $nonce_field;
	}

	/**
	 * Handle ESI request
	 *
	 */
	public static function hook_esi( $params ) {
		$form_id = $params[ 'form_id' ] ;
		echo self::nonce_field( $form_id, true ) ;
		exit ;
	}

	/**
	 * Create nonce action with form ID attatched
	 *
	 * @since 1.5.0
	 *
	 * @param string $form_id Form ID
	 *
	 * @return string
	 */
	protected static function nonce_action( $form_id ){
		return self::$action . $form_id;
	}

}